/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.classfinder;

import generic.jar.ResourceFile;
import generic.json.Json;
import ghidra.GhidraClassLoader;
import ghidra.framework.Application;
import ghidra.util.Msg;
import ghidra.util.Swing;
import ghidra.util.SystemUtilities;
import ghidra.util.classfinder.ClassDir;
import ghidra.util.classfinder.ClassFileInfo;
import ghidra.util.classfinder.ClassFilter;
import ghidra.util.classfinder.ClassJar;
import ghidra.util.classfinder.ExtensionPoint;
import ghidra.util.classfinder.ExtensionPointProperties;
import ghidra.util.datastruct.WeakDataStructureFactory;
import ghidra.util.datastruct.WeakSet;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.extensions.ExtensionDetails;
import ghidra.util.extensions.ExtensionModuleClassLoader;
import ghidra.util.extensions.ExtensionUtils;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import javax.swing.event.ChangeListener;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utilities.util.FileUtilities;
import utility.module.ModuleUtilities;

public class ClassSearcher {
    private static final Logger log = LogManager.getLogger(ClassSearcher.class);
    public static final String SEARCH_ALL_JARS_PROPERTY = "class.searcher.search.all.jars";
    static final boolean SEARCH_ALL_JARS = Boolean.getBoolean("class.searcher.search.all.jars");
    private static final boolean IS_USING_RESTRICTED_EXTENSIONS = Boolean.getBoolean("ghidra.extensions.classpath.restricted");
    private static List<Class<?>> FILTER_CLASSES = Arrays.asList(ExtensionPoint.class);
    private static Pattern extensionPointSuffixPattern;
    private static Map<String, Set<ClassFileInfo>> extensionPointSuffixToInfoMap;
    private static Map<ClassFileInfo, Class<?>> loadedCache;
    private static Set<ClassFileInfo> falsePositiveCache;
    private static volatile boolean hasSearched;
    private static volatile boolean isSearching;
    private static WeakSet<ChangeListener> listenerList;

    private ClassSearcher() {
    }

    public static void search(TaskMonitor monitor) throws CancelledException {
        if (hasSearched) {
            log.trace("Already searched for classes: using cached results");
            return;
        }
        log.trace("Using restricted extension class loader? " + IS_USING_RESTRICTED_EXTENSIONS);
        Instant start = Instant.now();
        isSearching = true;
        if (Application.inSingleJarMode()) {
            log.trace("Single Jar Mode: using extensions from the jar file");
            extensionPointSuffixToInfoMap = ClassSearcher.loadExtensionClassesFromJar();
        } else {
            extensionPointSuffixToInfoMap = ClassSearcher.findClasses(monitor);
        }
        log.trace("Found extension classes {}", extensionPointSuffixToInfoMap);
        if (extensionPointSuffixToInfoMap.isEmpty()) {
            throw new AssertException("Unable to locate extension points!");
        }
        hasSearched = true;
        isSearching = false;
        Swing.runNow(() -> ClassSearcher.fireClassListChanged());
        String finishedMessage = "Class search complete (" + ChronoUnit.MILLIS.between(start, Instant.now()) + " ms)";
        monitor.setMessage(finishedMessage);
        log.info(finishedMessage);
    }

    public static <T> List<Class<? extends T>> getClasses(Class<T> ancestorClass) {
        return ClassSearcher.getClasses(ancestorClass, null);
    }

    public static <T> List<Class<? extends T>> getClasses(Class<T> ancestorClass, Predicate<Class<? extends T>> classFilter) {
        if (!hasSearched) {
            return List.of();
        }
        if (isSearching) {
            throw new IllegalStateException("Cannot call the getClasses() while the ClassSearcher is searching!");
        }
        String suffix = ClassSearcher.getExtensionPointSuffix(ancestorClass.getName());
        if (suffix == null) {
            return List.of();
        }
        ArrayList<Class<T>> list = new ArrayList<Class<T>>();
        for (ClassFileInfo info : extensionPointSuffixToInfoMap.getOrDefault(suffix, Set.of())) {
            if (falsePositiveCache.contains(info)) continue;
            Class<?> c = loadedCache.get(info);
            if (c == null && (c = ClassSearcher.loadExtensionPoint(info.path(), info.name())) == null) {
                falsePositiveCache.add(info);
                continue;
            }
            loadedCache.put(info, c);
            if (!ancestorClass.isAssignableFrom(c) || classFilter != null && !classFilter.test(c)) continue;
            list.add(c);
        }
        ClassSearcher.prioritizeClasses(list);
        return list;
    }

    public static <T> List<T> getInstances(Class<T> c) {
        return ClassSearcher.getInstances(c, filter -> true);
    }

    public static <T> List<T> getInstances(Class<T> c, ClassFilter filter) {
        List<Class<T>> classes = ClassSearcher.getClasses(c);
        ArrayList<T> instances = new ArrayList<T>();
        for (Class<T> clazz : classes) {
            if (!filter.accepts(clazz)) continue;
            try {
                Constructor<T> constructor = clazz.getConstructor(null);
                T t = constructor.newInstance(null);
                instances.add(t);
            }
            catch (InstantiationException e) {
                Msg.showError(ClassSearcher.class, null, (String)"Error Instantiating Extension Point", (Object)("Error creating class " + clazz.getSimpleName() + " for extension " + c.getName() + ".  Discovered class is not a concrete implementation or does not have a nullary constructor!"), (Throwable)e);
            }
            catch (IllegalAccessException e) {
                Msg.showError(ClassSearcher.class, null, (String)"Error Instantiating Extension Point", (Object)("Error creating class " + clazz.getSimpleName() + " for extension " + c.getName() + ".  Discovered class does not have a public, default constructor!"), (Throwable)e);
            }
            catch (SecurityException e) {
                String message = "Error creating class " + clazz.getSimpleName() + " for extension " + c.getName() + ".  Security Exception!";
                Msg.showError(ClassSearcher.class, null, (String)"Error Instantiating Extension Point", (Object)message, (Throwable)e);
                throw new AssertException(message, (Throwable)e);
            }
            catch (Exception e) {
                Msg.showError(ClassSearcher.class, null, (String)"Error Creating Extension Point", (Object)("Error creating class " + clazz.getSimpleName() + " when creating extension points for " + c.getName()), (Throwable)e);
            }
        }
        return instances;
    }

    public static void addChangeListener(ChangeListener l) {
        listenerList.add(l);
    }

    public static void removeChangeListener(ChangeListener l) {
        listenerList.remove(l);
    }

    public static Set<ClassFileInfo> getExtensionPointInfo() {
        return extensionPointSuffixToInfoMap.values().stream().flatMap(e -> e.stream()).collect(Collectors.toSet());
    }

    public static Set<ClassFileInfo> getLoaded() {
        return loadedCache.keySet();
    }

    public static Set<ClassFileInfo> getFalsePositives() {
        return falsePositiveCache;
    }

    public static String getExtensionPointSuffix(String className) {
        Matcher m;
        if (extensionPointSuffixPattern == null) {
            extensionPointSuffixPattern = ClassSearcher.loadExtensionPointSuffixes();
        }
        if (className.contains("$") || className.endsWith("Test")) {
            return null;
        }
        int packageIndex = className.lastIndexOf(46);
        if (packageIndex > 0) {
            className = className.substring(packageIndex + 1);
        }
        return (m = extensionPointSuffixPattern.matcher(className)).find() && m.groupCount() == 1 ? m.group(1) : null;
    }

    public static boolean isClassOfInterest(Class<?> c) {
        if (Modifier.isAbstract(c.getModifiers())) {
            return false;
        }
        if (c.getEnclosingClass() != null) {
            return false;
        }
        if (!Modifier.isPublic(c.getModifiers())) {
            return false;
        }
        if (ExtensionPointProperties.Util.isExcluded(c)) {
            return false;
        }
        for (Class<?> filterClass : FILTER_CLASSES) {
            if (!filterClass.isAssignableFrom(c)) continue;
            return true;
        }
        return false;
    }

    public static void logStatistics() {
        log.info("Class searcher loaded %d extension points (%d false positives)".formatted(loadedCache.size(), falsePositiveCache.size()));
    }

    private static Map<String, Set<ClassFileInfo>> findClasses(TaskMonitor monitor) throws CancelledException {
        log.info("Searching for classes...");
        ArrayList<ClassDir> classDirs = new ArrayList<ClassDir>();
        ArrayList<ClassJar> classJars = new ArrayList<ClassJar>();
        for (String string : ClassSearcher.gatherSearchPaths()) {
            String lcSearchPath = string.toLowerCase();
            File searchFile = new File(string);
            if ((lcSearchPath.endsWith(".jar") || lcSearchPath.endsWith(".zip")) && searchFile.exists()) {
                if (ClassJar.ignoreJar(string)) {
                    log.trace("Ignoring jar file: {}", (Object)string);
                    continue;
                }
                log.trace("Searching jar file: {}", (Object)string);
                classJars.add(new ClassJar(string, monitor));
                continue;
            }
            if (!searchFile.isDirectory()) continue;
            log.trace("Searching classpath directory: {}", (Object)string);
            classDirs.add(new ClassDir(string, monitor));
        }
        ArrayList<ClassFileInfo> classList = new ArrayList<ClassFileInfo>();
        for (ClassDir dir : classDirs) {
            monitor.checkCancelled();
            dir.getClasses(classList, monitor);
        }
        for (ClassJar jar : classJars) {
            monitor.checkCancelled();
            jar.getClasses(classList, monitor);
        }
        HashMap<String, ClassFileInfo> hashMap = new HashMap<String, ClassFileInfo>();
        for (ClassFileInfo info : classList) {
            ClassFileInfo existing = (ClassFileInfo)hashMap.get(info.name());
            if (existing == null) {
                hashMap.put(info.name(), info);
                continue;
            }
            log.info("Ignoring class '%s' from '%s'. Already found at '%s'.".formatted(info.name(), info.path(), existing.path()));
        }
        return hashMap.values().stream().collect(Collectors.groupingBy(ClassFileInfo::suffix, Collectors.toSet()));
    }

    private static <T> void prioritizeClasses(List<Class<? extends T>> list) {
        Collections.sort(list, (c1, c2) -> {
            int p2;
            int p1 = ExtensionPointProperties.Util.getPriority(c1);
            if (p1 > (p2 = ExtensionPointProperties.Util.getPriority(c2))) {
                return -1;
            }
            if (p1 < p2) {
                return 1;
            }
            return c1.getName().compareTo(c2.getName());
        });
    }

    private static List<String> gatherSearchPaths() {
        ArrayList<String> rawPaths = new ArrayList<String>();
        rawPaths.addAll(GhidraClassLoader.getClasspath((String)"java.class.path"));
        rawPaths.addAll(GhidraClassLoader.getClasspath((String)"java.class.path.ext"));
        return ClassSearcher.canonicalizePaths(rawPaths);
    }

    private static List<String> canonicalizePaths(Collection<String> paths) {
        List<String> canonical = paths.stream().map(path -> {
            String normalized = ClassSearcher.normalize(path);
            return normalized;
        }).collect(Collectors.toList());
        return canonical;
    }

    private static String normalize(String path) {
        try {
            Path p = Paths.get(path, new String[0]);
            Path normalized = p.normalize();
            Path absolutePath = normalized.toAbsolutePath();
            return absolutePath.toString();
        }
        catch (InvalidPathException e) {
            log.trace("Invalid path '{}'", (Object)path);
            return path;
        }
    }

    private static Map<String, Set<ClassFileInfo>> loadExtensionClassesFromJar() {
        ResourceFile appRoot = Application.getApplicationRootDirectory();
        ResourceFile extensionClassesFile = new ResourceFile(appRoot, "EXTENSION_POINT_CLASSES");
        try {
            List classNames = FileUtilities.getLines((ResourceFile)extensionClassesFile);
            HashSet<ClassFileInfo> extensionClasses = new HashSet<ClassFileInfo>();
            for (String className : classNames) {
                String epName = ClassSearcher.getExtensionPointSuffix(className);
                if (epName == null) continue;
                extensionClasses.add(new ClassFileInfo(appRoot.getAbsolutePath(), className, epName));
            }
            return extensionClasses.stream().collect(Collectors.groupingBy(ClassFileInfo::suffix, Collectors.toSet()));
        }
        catch (IOException e) {
            throw new AssertException("Unexpected IOException reading extension class file " + String.valueOf(extensionClassesFile), (Throwable)e);
        }
    }

    private static Pattern loadExtensionPointSuffixes() {
        HashSet<String> extensionPointSuffixes = new HashSet<String>();
        Collection<ResourceFile> moduleRootDirectories = Application.getModuleRootDirectories();
        if (moduleRootDirectories.isEmpty()) {
            throw new AssertException("Could not find modules for Class Searcher!");
        }
        log.trace("Scanning module root directories: {}", moduleRootDirectories);
        for (ResourceFile moduleRoot : moduleRootDirectories) {
            ResourceFile file = new ResourceFile(moduleRoot, "data/ExtensionPoint.manifest");
            if (!file.exists()) continue;
            for (String line : FileUtilities.getLinesQuietly((ResourceFile)file)) {
                line = line.trim();
                try {
                    Pattern.compile(line);
                    extensionPointSuffixes.add(line);
                }
                catch (PatternSyntaxException e) {
                    throw new AssertException("Error parsing extension point suffix '%s' found in '%s'".formatted(line, file));
                }
            }
        }
        StringBuilder buffy = new StringBuilder(".*(");
        String between = "";
        for (String suffix : extensionPointSuffixes) {
            if ((suffix = suffix.trim()).isEmpty()) continue;
            buffy.append(between);
            buffy.append(suffix);
            between = "|";
        }
        buffy.append(")$");
        log.trace("Using extension point pattern: {}", (Object)buffy);
        return Pattern.compile(buffy.toString());
    }

    private static void fireClassListChanged() {
        for (ChangeListener listener : listenerList) {
            try {
                listener.stateChanged(null);
            }
            catch (Throwable t) {
                Msg.showError(ClassSearcher.class, null, (String)"Exception", (Object)"Error in listener for class list changed", (Throwable)t);
            }
        }
    }

    private static Class<?> loadExtensionPoint(String path, String className) {
        if (ClassSearcher.getExtensionPointSuffix(className) == null) {
            return null;
        }
        ClassLoader classLoader = ClassSearcher.getClassLoader(path);
        try {
            Class<?> c = Class.forName(className, true, classLoader);
            if (ClassSearcher.isClassOfInterest(c)) {
                return c;
            }
        }
        catch (Throwable t) {
            ClassSearcher.processClassLoadError(path, className, t);
        }
        return null;
    }

    private static ClassLoader getClassLoader(String path) {
        ClassLoader classLoader = ClassSearcher.class.getClassLoader();
        if (!IS_USING_RESTRICTED_EXTENSIONS) {
            return classLoader;
        }
        ExtensionDetails extension = ExtensionUtils.getExtension(path);
        if (extension != null) {
            log.trace(() -> "Installing custom extension class loader for: " + Json.toStringFlat(extension));
            classLoader = new ExtensionModuleClassLoader(extension);
        }
        return classLoader;
    }

    private static void processClassLoadError(String path, String name, Throwable t) {
        if (t instanceof LinkageError) {
            log.trace("LinkageError loading class {}; Incompatible class version? ", (Object)name, (Object)t);
            return;
        }
        if (!(t instanceof ClassNotFoundException)) {
            log.error("Error loading class {} - {}", (Object)name, (Object)t.getMessage(), (Object)t);
            return;
        }
        ClassSearcher.processClassNotFoundExcepetion(path, name, (ClassNotFoundException)t);
    }

    private static void processClassNotFoundExcepetion(String path, String name, ClassNotFoundException t) {
        if (!ClassSearcher.isModuleEntryMissingFromClasspath(path)) {
            log.error("Error loading class {} - {}", (Object)name, (Object)t.getMessage(), (Object)t);
            return;
        }
        if (SystemUtilities.isInTestingMode()) {
            return;
        }
        log.error("Module class is missing from the classpath.\n\tUpdate your launcher accordingly.\n\tModule: '" + path + "'\n\tClass: '" + name + "'");
    }

    private static boolean isModuleEntryMissingFromClasspath(String path) {
        boolean inModule = ModuleUtilities.isInModule((String)path);
        if (!inModule) {
            return false;
        }
        String classPath = System.getProperty("java.class.path");
        boolean inClassPath = classPath.contains(path);
        return !inClassPath;
    }

    static {
        loadedCache = new HashMap();
        falsePositiveCache = new HashSet<ClassFileInfo>();
        listenerList = WeakDataStructureFactory.createCopyOnReadWeakSet();
    }
}

