/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.memory;

import docking.widgets.OptionDialog;
import docking.widgets.dialogs.NumberInputDialog;
import docking.widgets.table.AbstractSortedTableModel;
import ghidra.app.plugin.core.memory.MemoryMapProvider;
import ghidra.app.plugin.core.memory.UninitializedBlockCmd;
import ghidra.framework.cmd.BackgroundCommand;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainObject;
import ghidra.framework.store.LockException;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressRange;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.address.OverlayAddressSpace;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.mem.MemoryBlockSourceInfo;
import ghidra.program.model.mem.MemoryBlockType;
import ghidra.program.util.ProgramLocation;
import ghidra.program.util.ProgramSelection;
import ghidra.util.Msg;
import ghidra.util.Swing;
import ghidra.util.table.ProgramTableModel;
import java.awt.Component;
import java.awt.Cursor;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.stream.Collectors;
import javax.swing.JTable;
import javax.swing.table.TableCellEditor;
import org.apache.commons.lang3.exception.ExceptionUtils;

class MemoryMapModel
extends AbstractSortedTableModel<MemoryBlock>
implements ProgramTableModel {
    static final byte NAME = 0;
    static final byte START = 1;
    static final byte END = 2;
    static final byte LENGTH = 3;
    static final byte READ = 4;
    static final byte WRITE = 5;
    static final byte EXECUTE = 6;
    static final byte VOLATILE = 7;
    static final byte ARTIFICIAL = 8;
    static final byte OVERLAY = 9;
    static final byte BLOCK_TYPE = 10;
    static final byte INIT = 11;
    static final byte BYTE_SOURCE = 12;
    static final byte SOURCE = 13;
    static final byte COMMENT = 14;
    static final String NAME_COL = "Name";
    static final String START_COL = "Start";
    static final String END_COL = "End";
    static final String LENGTH_COL = "Length";
    static final String READ_COL = "R";
    static final String WRITE_COL = "W";
    static final String EXECUTE_COL = "X";
    static final String VOLATILE_COL = "Volatile";
    static final String ARTIFICIAL_COL = "Artificial";
    static final String OVERLAY_COL = "Overlayed Space";
    static final String BLOCK_TYPE_COL = "Type";
    static final String INIT_COL = "Initialized";
    static final String BYTE_SOURCE_COL = "Byte Source";
    static final String SOURCE_COL = "Source";
    static final String COMMENT_COL = "Comment";
    private static final Cursor WAIT_CURSOR = Cursor.getPredefinedCursor(3);
    private static final Cursor DEFAULT_CURSOR = Cursor.getPredefinedCursor(0);
    private Program program;
    private List<MemoryBlock> memList;
    private MemoryMapProvider provider;
    private static final String[] COLUMN_NAMES = new String[]{"Name", "Start", "End", "Length", "R", "W", "X", "Volatile", "Artificial", "Overlayed Space", "Type", "Initialized", "Byte Source", "Source", "Comment"};

    MemoryMapModel(MemoryMapProvider provider, Program program) {
        super(1);
        this.provider = provider;
        this.setProgram(program);
    }

    void setProgram(Program program) {
        this.program = program;
        this.populateMap();
    }

    private void populateMap() {
        MemoryBlock[] blocks;
        this.memList = new ArrayList<MemoryBlock>();
        if (this.program == null) {
            this.fireTableDataChanged();
            return;
        }
        Memory mem = this.program.getMemory();
        for (MemoryBlock block : blocks = mem.getBlocks()) {
            this.memList.add(block);
        }
        this.fireTableDataChanged();
    }

    void update() {
        JTable table = this.provider.getTable();
        TableCellEditor cellEditor = table.getCellEditor();
        if (cellEditor != null) {
            cellEditor.cancelCellEditing();
        }
        this.populateMap();
    }

    public boolean isSortable(int columnIndex) {
        return columnIndex != 4 && columnIndex != 5 && columnIndex != 6 && columnIndex != 7 && columnIndex != 8 && columnIndex != 11;
    }

    public String getName() {
        return "Memory Map";
    }

    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    public String getColumnName(int column) {
        if (column < 0 || column >= COLUMN_NAMES.length) {
            return "UNKNOWN";
        }
        return COLUMN_NAMES[column];
    }

    public int findColumn(String columnName) {
        for (int i = 0; i < COLUMN_NAMES.length; ++i) {
            if (!COLUMN_NAMES[i].equals(columnName)) continue;
            return i;
        }
        return 0;
    }

    public Class<?> getColumnClass(int columnIndex) {
        if (columnIndex == 4 || columnIndex == 5 || columnIndex == 6 || columnIndex == 7 || columnIndex == 8 || columnIndex == 11) {
            return Boolean.class;
        }
        return String.class;
    }

    public boolean isCellEditable(int rowIndex, int columnIndex) {
        switch (columnIndex) {
            case 0: 
            case 4: 
            case 5: 
            case 6: 
            case 7: 
            case 8: {
                return true;
            }
            case 14: {
                return true;
            }
            case 11: {
                MemoryBlock block = this.memList.get(rowIndex);
                MemoryBlockType blockType = block.getType();
                if (blockType == MemoryBlockType.BIT_MAPPED || blockType == MemoryBlockType.BYTE_MAPPED) break;
                return true;
            }
        }
        return false;
    }

    public int getRowCount() {
        return this.memList.size();
    }

    private String getAddressString(Address address) {
        return address.toString();
    }

    public MemoryBlock getBlockAt(int rowIndex) {
        if (this.memList == null) {
            return null;
        }
        if (rowIndex < 0 || rowIndex >= this.memList.size()) {
            return null;
        }
        return this.memList.get(rowIndex);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setValueAt(Object aValue, int row, int column) {
        this.provider.setCursor(WAIT_CURSOR);
        try {
            MemoryBlock block = this.getBlockAt(row);
            if (block == null) {
                return;
            }
            this.doSetValueAt(aValue, row, column, block);
        }
        finally {
            this.provider.setCursor(DEFAULT_CURSOR);
        }
    }

    private void doSetValueAt(Object aValue, int row, int column, MemoryBlock block) {
        switch (column) {
            case 0: {
                this.setName(block, (String)aValue);
                break;
            }
            case 4: {
                this.program.withTransaction("Set Read State", () -> {
                    boolean value = (Boolean)aValue;
                    block.setRead(value);
                    this.provider.setStatusText("");
                });
                break;
            }
            case 5: {
                this.program.withTransaction("Set Write State", () -> {
                    boolean value = (Boolean)aValue;
                    block.setWrite(value);
                    this.provider.setStatusText("");
                });
                break;
            }
            case 6: {
                this.program.withTransaction("Set Execute State", () -> {
                    boolean value = (Boolean)aValue;
                    block.setExecute(value);
                    this.provider.setStatusText("");
                });
                break;
            }
            case 7: {
                this.program.withTransaction("Set Volatile State", () -> {
                    boolean value = (Boolean)aValue;
                    block.setVolatile(value);
                    this.provider.setStatusText("");
                });
                break;
            }
            case 8: {
                this.program.withTransaction("Set Artificial State", () -> {
                    boolean value = (Boolean)aValue;
                    block.setArtificial(value);
                    this.provider.setStatusText("");
                });
                break;
            }
            case 11: {
                MemoryBlockType blockType = block.getType();
                if (blockType == MemoryBlockType.BIT_MAPPED || blockType == MemoryBlockType.BYTE_MAPPED) {
                    this.showMessage("Cannot change intialized memory state of a mapped Block");
                    break;
                }
                this.provider.setStatusText("");
                boolean booleanValue = (Boolean)aValue;
                if (booleanValue) {
                    this.initializeBlock(block);
                    break;
                }
                this.revertBlockToUninitialized(block);
                break;
            }
            case 13: {
                break;
            }
            case 14: {
                this.setComment(block, (String)aValue);
                break;
            }
        }
        this.fireTableRowsUpdated(row, column);
    }

    private void setComment(MemoryBlock block, String aValue) {
        String cmt = block.getComment();
        if (cmt == null || !cmt.equals(aValue)) {
            if (aValue.length() == 0) {
                aValue = null;
            }
            String newValue = aValue;
            this.program.withTransaction("Set Comment", () -> block.setComment(newValue));
        }
    }

    private void setName(MemoryBlock block, String name) {
        if (!this.verifyRenameAllowed(block, name = name.trim())) {
            return;
        }
        if (name.length() == 0) {
            Msg.showError((Object)this, (Component)this.provider.getComponent(), (String)"Enter Block Label", (Object)"Please enter a label name.");
            return;
        }
        if (name.equals(block.getName())) {
            return;
        }
        if (!Memory.isValidMemoryBlockName((String)name)) {
            Msg.showError((Object)this, (Component)this.provider.getComponent(), (String)"Invalid Name", (Object)("Invalid Memory Block Name: " + name));
            return;
        }
        try {
            String newName = name;
            this.program.withTransaction("Rename Memory Block", () -> block.setName(newName));
        }
        catch (LockException e) {
            this.provider.setStatusText(e.getMessage());
        }
    }

    private void revertBlockToUninitialized(MemoryBlock block) {
        int result = OptionDialog.showYesNoDialog((Component)this.provider.getComponent(), (String)"Confirm Setting Block To Uninitialized", (String)"Are you sure you want to remove the bytes from this block? \n\nThis will result in removing all functions, instructions, data,\nand outgoing references from the block!");
        if (result == 2) {
            return;
        }
        UninitializedBlockCmd cmd = new UninitializedBlockCmd(block);
        this.provider.getTool().executeBackgroundCommand((BackgroundCommand)cmd, (DomainObject)this.program);
    }

    private boolean verifyRenameAllowed(MemoryBlock block, String newName) {
        if (!block.isOverlay() || block.getName().equals(newName)) {
            return true;
        }
        if (!this.program.hasExclusiveAccess()) {
            String msg = "Close the file and undo your checkout,\nthen do a checkout with the exclusive lock.";
            DomainFile df = this.program.getDomainFile();
            if (df.modifiedSinceCheckout() || df.isChanged()) {
                msg = "Check in this file, then do a checkout with the\nexclusive lock.";
            }
            Msg.showInfo(this.getClass(), (Component)this.provider.getComponent(), (String)"Exclusive Checkout Required", (Object)("An exclusive checkout is required in order to\nrename an overlay memory block.\n" + msg));
            return false;
        }
        return true;
    }

    private void initializeBlock(MemoryBlock block) {
        NumberInputDialog dialog = new NumberInputDialog("Initialize Memory Block", "Enter fill byte value for block: ", Integer.valueOf(0), 0, 255, true);
        if (!dialog.show()) {
            return;
        }
        byte value = (byte)dialog.getValue();
        int id = this.program.startTransaction("Initialize Memory Block");
        try {
            Memory mem = this.program.getMemory();
            int index = this.memList.indexOf(block);
            MemoryBlock newBlock = mem.convertToInitialized(block, value);
            this.memList.set(index, newBlock);
            this.program.endTransaction(id, true);
        }
        catch (Throwable t) {
            this.program.endTransaction(id, false);
            String msg = ExceptionUtils.getMessage((Throwable)t);
            Msg.showError((Object)this, (Component)this.provider.getComponent(), (String)"Block Initialization Failed", (Object)msg, (Throwable)t);
        }
    }

    private void showMessage(String msg) {
        Swing.runLater(() -> this.provider.setStatusText(msg));
    }

    public Object getColumnValueForRow(MemoryBlock block, int columnIndex) {
        try {
            switch (columnIndex) {
                case 0: {
                    return block.getName();
                }
                case 1: {
                    return this.getAddressString(block.getStart());
                }
                case 2: {
                    return this.getAddressString(block.getEnd());
                }
                case 3: {
                    long len = block.getEnd().subtract(block.getStart()) + 1L;
                    return "0x" + Long.toHexString(len);
                }
                case 4: {
                    return block.isRead() ? Boolean.TRUE : Boolean.FALSE;
                }
                case 5: {
                    return block.isWrite() ? Boolean.TRUE : Boolean.FALSE;
                }
                case 6: {
                    return block.isExecute() ? Boolean.TRUE : Boolean.FALSE;
                }
                case 7: {
                    return block.isVolatile() ? Boolean.TRUE : Boolean.FALSE;
                }
                case 8: {
                    return block.isArtificial() ? Boolean.TRUE : Boolean.FALSE;
                }
                case 9: {
                    return this.getOverlayBaseSpaceName(block);
                }
                case 11: {
                    MemoryBlockType blockType = block.getType();
                    if (blockType == MemoryBlockType.BIT_MAPPED) {
                        return null;
                    }
                    return block.isInitialized() ? Boolean.TRUE : Boolean.FALSE;
                }
                case 12: {
                    return this.getByteSourceDescription(block.getSourceInfos());
                }
                case 13: {
                    if (block.getType() == MemoryBlockType.BIT_MAPPED || block.getType() == MemoryBlockType.BYTE_MAPPED) {
                        MemoryBlockSourceInfo info = (MemoryBlockSourceInfo)block.getSourceInfos().get(0);
                        return ((AddressRange)info.getMappedRange().get()).getMinAddress().toString();
                    }
                    return block.getSourceName();
                }
                case 14: {
                    return block.getComment();
                }
                case 10: {
                    return block.getType().toString();
                }
            }
            return "UNKNOWN";
        }
        catch (ConcurrentModificationException e) {
            this.update();
            return null;
        }
    }

    private String getByteSourceDescription(List<MemoryBlockSourceInfo> sourceInfos) {
        List<MemoryBlockSourceInfo> limited = sourceInfos.size() < 5 ? sourceInfos : sourceInfos.subList(0, 4);
        Object description = limited.stream().map(info -> info.getDescription()).collect(Collectors.joining(" | "));
        if (limited != sourceInfos) {
            description = (String)description + "...";
        }
        return description;
    }

    public List<MemoryBlock> getModelData() {
        return this.memList;
    }

    protected Comparator<MemoryBlock> createSortComparator(int columnIndex) {
        if (columnIndex == 12) {
            return super.createSortComparator(columnIndex);
        }
        return new MemoryMapComparator(columnIndex);
    }

    @Override
    public ProgramLocation getProgramLocation(int row, int column) {
        MemoryBlock block = (MemoryBlock)this.getRowObject(row);
        Address address = block.getStart();
        if (column == 2) {
            address = block.getEnd();
        }
        return new ProgramLocation(this.program, address);
    }

    @Override
    public ProgramSelection getProgramSelection(int[] rows) {
        if (rows.length == 0) {
            return null;
        }
        AddressSet addressSet = new AddressSet();
        for (int row : rows) {
            MemoryBlock block = (MemoryBlock)this.getRowObject(row);
            Address start = block.getStart();
            Address end = block.getEnd();
            if (!start.isMemoryAddress() || !end.isMemoryAddress()) continue;
            addressSet.addRange(start, end);
        }
        return new ProgramSelection((AddressSetView)addressSet);
    }

    @Override
    public Program getProgram() {
        return this.program;
    }

    private String getOverlayBaseSpaceName(MemoryBlock block) {
        AddressSpace space = block.getStart().getAddressSpace();
        if (space instanceof OverlayAddressSpace) {
            OverlayAddressSpace ovSpace = (OverlayAddressSpace)space;
            return ovSpace.getOverlayedSpace().getName();
        }
        return "";
    }

    private class MemoryMapComparator
    implements Comparator<MemoryBlock> {
        private final int sortColumn;

        public MemoryMapComparator(int sortColumn) {
            this.sortColumn = sortColumn;
        }

        @Override
        public int compare(MemoryBlock b1, MemoryBlock b2) {
            switch (this.sortColumn) {
                case 0: {
                    return b1.getName().compareToIgnoreCase(b2.getName());
                }
                case 1: {
                    return b1.getStart().compareTo((Object)b2.getStart());
                }
                case 2: {
                    return b1.getEnd().compareTo((Object)b2.getEnd());
                }
                case 3: {
                    return (int)(b1.getSize() - b2.getSize());
                }
                case 4: {
                    return Boolean.compare(b1.isRead(), b2.isRead());
                }
                case 5: {
                    return Boolean.compare(b1.isWrite(), b2.isWrite());
                }
                case 6: {
                    return Boolean.compare(b1.isExecute(), b2.isExecute());
                }
                case 7: {
                    return Boolean.compare(b1.isVolatile(), b2.isVolatile());
                }
                case 8: {
                    return Boolean.compare(b1.isArtificial(), b2.isArtificial());
                }
                case 9: {
                    String ov1 = MemoryMapModel.this.getOverlayBaseSpaceName(b1);
                    String ov2 = MemoryMapModel.this.getOverlayBaseSpaceName(b2);
                    return ov1.compareTo(ov2);
                }
                case 11: {
                    return Boolean.compare(b1.isInitialized(), b2.isInitialized());
                }
                case 13: {
                    String b1src = b1.getSourceName();
                    String b2src = b2.getSourceName();
                    if (b1src == null) {
                        b1src = "";
                    }
                    if (b2src == null) {
                        b2src = "";
                    }
                    return b1src.compareToIgnoreCase(b2src);
                }
                case 14: {
                    String comment1 = b1.getComment();
                    String comment2 = b2.getComment();
                    if (comment1 == null) {
                        comment1 = "";
                    }
                    if (comment2 == null) {
                        comment2 = "";
                    }
                    return comment1.compareToIgnoreCase(comment2);
                }
                case 10: {
                    String bt1 = b1.getType().toString();
                    String bt2 = b2.getType().toString();
                    return bt1.compareToIgnoreCase(bt2);
                }
            }
            throw new RuntimeException("Unimplemented column comparator: " + this.sortColumn);
        }
    }
}

